package gov.va.med.domain.service.messaging.impl;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import gov.va.med.domain.model.RxRefillPayload;
import gov.va.med.domain.service.messaging.DestinedRequest;
import gov.va.med.domain.service.messaging.IMessageOriginator;
import gov.va.med.domain.service.messaging.IMessageSender;
import gov.va.med.domain.service.messaging.MessagingConstants;
import gov.va.med.domain.service.messaging.Request;
import gov.va.med.domain.service.messaging.Response;
import gov.va.med.domain.service.messaging.environment.DestinationKey;

/**
 * Facade to the business tier to simplfiy the invocation of predefined requests and encapsulate the
 * messaging-specific logic.
 * <P>
 * A method is provided for each type of supported request and new methods will be added as new
 * requests are implemented. <BR>
 * NOTE: Each added method must be also implemented in IMessageOriginator.
 * <P>
 * There is no public constructor because this class must be loaded from the Spring Framework's Bean
 * Factory for proper transaction management.
 * <P>
 * To obtain an instance of this class, use the static method <BR>
 * public static IMessageOriginator getMessageOriginator()
 * <P>
 * 
 * @author Joel Goldberg
 * @version $Id: MessageOriginator.java,v 1.38 2005/08/27 19:19:20 alex.kalinovsky Exp $
 * @since MHV 2.0 <br>
 *        Apr 17, 2005
 */
@Component
public class MessageOriginator implements MessagingConstants, IMessageOriginator {

   @Autowired
   private IMessageSender messageSender;
	

   /**
    * This method is public solely to allow Spring Bean Factory to create it. No application code
    * should use this method directly.
    * <P>
    * 
    * Use <code> public static IMessageOriginator getMessageOriginator() </code>
    * 
    */
   public MessageOriginator() {
   }

   /**
    * Make a synchronous call to a VistA site to for this RxRefillRequest request.
    * <UL>
    * Known functions to date (although not restricted to these) are:
    * <LI>refillPrescriptions
    * <LI>getPrescriptionStatus
    * <LI>getPrescriptionProfile
    * </UL>
    * All functions expect an RxRefillPayload.
    * 
    * @param request containing an RxRefillPayload containing the patient on whose behalf the
    *           request is being made, and the institution to which the request will be sent.
    * 
    * @return Response containing RxRefillPayload or StatusPayload, or ErrorResponse
    */
   public Response sendRxRefillRequest(Request request) {

      RxRefillPayload rxRefillPayload = (RxRefillPayload) request.getPayload();
      DestinationKey key = new DestinationKey(rxRefillPayload.getInstitution().getStationNumber());
      key.setInstitutionId(new Long(rxRefillPayload.getInstitution().getLongId()));

      DestinedRequest destinedRequest = new DestinedRequest(key, request.getMessageType(), rxRefillPayload);
      destinedRequest.setUserId(request.getUserId());
      return sendSynchronousMessage(destinedRequest);
   }

   /**
    * Reusable method to send any DestinedRequest synchronously
    * 
    * @param request as needed for your purposes.
    * @return Response containing RxRefillPayload or StatusPayload, or ErrorResponse
    */
   public Response sendSynchronousMessage(DestinedRequest request) {
      return messageSender.sendSynchronousMessage(request);
   }
   
}
